/*---------------------------------------------------------------------------*\

    FILE....: MULAW.C
    TYPE....: C Functions
    AUTHOR..: David Rowe
    DATE....: 16/10/97

    Mu-law compression functions.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>

#include "contypes.h"
#include "alawmulaw.h"

/*-------------------------------------------------------------------------*\

				DEFINES

\*-------------------------------------------------------------------------*/

/* mu-law constants */

#define	LSIGN	0x8000		/* linear sign bit 15 = 1 if negative	   */
#define	MUSIGN	0x80		/* mu-law sign bit 7 = 1 if negative	   */
#define	MUMAX	0x1fff		/* max mu-law value in linear form	   */
#define	MUMAG	0x7f		/* mu-law magnitude bits 		   */
#define	BIAS	33		/* converts seg end points to powers of 2  */
#define	LINEAR	0xf		/* masks of linear portion of magnitude	   */

/*-------------------------------------------------------------------------*\

			    FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: mulaw_encode
	AUTHOR...: David Rowe
	DATE.....: 14/10/97

	Encodes a vector of linear samples to mu-law equivalents.  The linear
	samples are assumed to be in Q15 format.

	Reference: TI databook
		   "Theory, Alogorithms, and Implementations Vol 1", p171

	Mu-law format: psssqqqq

		       p is the sign
		       s is the segment
		       q if the quantisation bin number

\*--------------------------------------------------------------------------*/

void mulaw_encode(char mulaw[], short linear[], USHORT sz)
/*  char   mulaw[];	mulaw encoded samples                           */
/*  word   linear[];	16 bit input samples	                        */
/*  USHORT sz;		number of sample in buffers			*/
{
    USHORT y,mag;
    USHORT p,s,q,b;
    ULONG  acc;
    int    i;

    for(i=0; i<sz; i++) {
	acc = linear[i] >> MULAW_SCALE;

	/* separate sign and bias */

	if (acc & LSIGN) {
	    p = MUSIGN;
	    mag = -acc;
	}
	else {
	    p = 0;
	    mag = acc;
	}
	mag += BIAS;

	/* encode magnitude */

	if (mag > MUMAX)
	    y = p + MUMAG;
	else {

	    /* determine left most bit and therefore segment */

	    b = 0;
	    acc = mag >> 1;
	    while(acc) {
		b++;
		acc >>= 1;
	    }
	    s = b - 5;

	    /* extract quantisation bin */

	    q = mag >> (b-4);
	    q &= LINEAR;
	    y = p + (s<<4) + q;
	}

	mulaw[i] = ~y & 0xff;
    }
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: mulaw_decode
	AUTHOR...: David Rowe
	DATE.....: 14/10/97

	Decodes a vector of mu-law samples to linear equivalents.  The linear
	samples are normalised to Q15.

	Reference: TI databook
		   "Theory, Alogorithms, and Implementations Vol 1", p171

\*--------------------------------------------------------------------------*/

void mulaw_decode(short linear[], char mulaw[], USHORT sz)
/*  word   linear[];	16 bit output samples           		*/
/*  char   mulaw[];	mulaw encoded samples                           */
/*  USHORT sz;		number of sample in buffers			*/
{
    ULONG  acc,p,s,q;
    int    i;

    for(i=0; i<sz; i++) {
	acc = ~mulaw[i] & 0xff;

	/* extract sign */

	if (acc & MUSIGN)
	    p = LSIGN;
	else
	    p = 0;
	acc &= MUMAG;

	/* extract q and s */

	q = acc & 0xf;
	s = acc >> 4;

	/* remove bias and form linear sample */

	acc = 2*q+BIAS;
	acc <<= s;
	acc -= BIAS;
	acc <<= MULAW_SCALE;
	if (p)
	    linear[i] = -acc;
	else
	    linear[i] = acc;
    }

}

